/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at <http://mozilla.org/MPL/2.0/>. */

// Tests tracing all function returns

"use strict";

add_task(async function testTracingFunctionReturn() {
  await pushPref("devtools.debugger.features.javascript-tracing", true);

  const jsCode = `async function foo() { nullReturn(); falseReturn(); await new Promise(r => setTimeout(r, 0)); return bar(); }; function nullReturn() { return null; } function falseReturn() { return false; } function bar() { return 42; }; function throwingFunction() { throw new Error("the exception") }`;
  const dbg = await initDebuggerWithAbsoluteURL(
    "data:text/html," +
      encodeURIComponent(`<script>${jsCode}</script><body></body>`)
  );

  // This test covers the Web Console, whereas it is no longer the default output
  await toggleJsTracerMenuItem(dbg, "#jstracer-menu-item-console");

  await toggleJsTracerMenuItem(dbg, "#jstracer-menu-item-function-return");

  info("Enable tracing with function returns, but without values");
  await toggleJsTracer(dbg.toolbox);

  invokeInTab("foo");
  await hasConsoleMessage(dbg, "⟶ interpreter λ foo");
  await hasConsoleMessage(dbg, "⟶ interpreter λ bar");
  await hasConsoleMessage(dbg, "⟵ λ bar");
  await hasConsoleMessage(dbg, "⟵ λ foo");

  await toggleJsTracer(dbg.toolbox);

  await toggleJsTracerMenuItem(dbg, "#jstracer-menu-item-log-values");

  info("Re-enable with returned values");
  await toggleJsTracer(dbg.toolbox);

  invokeInTab("foo");

  await hasConsoleMessage(dbg, "⟶ interpreter λ foo");
  await hasConsoleMessage(dbg, "⟶ interpreter λ bar");
  await hasConsoleMessage(dbg, "⟵ λ bar return 42");
  await hasConsoleMessage(dbg, "⟶ interpreter λ nullReturn");
  await hasConsoleMessage(dbg, "⟵ λ nullReturn return null");
  await hasConsoleMessage(dbg, "⟶ interpreter λ falseReturn");
  await hasConsoleMessage(dbg, "⟵ λ falseReturn return false");
  await hasConsoleMessage(
    dbg,
    `⟵ λ foo return \nPromise { <state>: "fulfilled", <value>: 42 }`
  );

  invokeInTab("throwingFunction").catch(() => {});
  await hasConsoleMessage(
    dbg,
    `⟵ λ throwingFunction throw \nError: the exception`
  );

  info("Stop tracing");
  await toggleJsTracer(dbg.toolbox);

  info("Toggle the two settings to the default value");
  await toggleJsTracerMenuItem(dbg, "#jstracer-menu-item-log-values");
  await toggleJsTracerMenuItem(dbg, "#jstracer-menu-item-function-return");

  // Reset the trace on next interaction setting
  Services.prefs.clearUserPref(
    "devtools.debugger.javascript-tracing-on-next-interaction"
  );
});
